-- نظام إدارة عيادة الأسنان - هيكل قاعدة البيانات
-- Dental Clinic Management System - Database Schema

-- إعداد قاعدة البيانات
SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET AUTOCOMMIT = 0;
START TRANSACTION;
SET time_zone = "+00:00";

-- ===================================
-- جدول المستخدمين (Users Table)
-- ===================================
CREATE TABLE `users` (
  `user_id` int(11) NOT NULL AUTO_INCREMENT,
  `username` varchar(50) NOT NULL,
  `email` varchar(100) NOT NULL,
  `password_hash` varchar(255) NOT NULL,
  `first_name` varchar(50) NOT NULL,
  `last_name` varchar(50) NOT NULL,
  `phone` varchar(20) DEFAULT NULL,
  `address` text DEFAULT NULL,
  `user_type` enum('admin','doctor','receptionist') NOT NULL,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  `last_login` timestamp NULL DEFAULT NULL,
  PRIMARY KEY (`user_id`),
  UNIQUE KEY `username` (`username`),
  UNIQUE KEY `email` (`email`),
  KEY `idx_user_type` (`user_type`),
  KEY `idx_is_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول الموظفين (Staff Table)
-- ===================================
CREATE TABLE `staff` (
  `staff_id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) DEFAULT NULL,
  `employee_id` varchar(20) NOT NULL,
  `first_name` varchar(50) NOT NULL,
  `last_name` varchar(50) NOT NULL,
  `date_of_birth` date DEFAULT NULL,
  `gender` enum('male','female') NOT NULL,
  `phone` varchar(20) NOT NULL,
  `email` varchar(100) DEFAULT NULL,
  `address` text DEFAULT NULL,
  `qualification` varchar(200) DEFAULT NULL,
  `specialization` varchar(100) DEFAULT NULL,
  `experience_years` int(11) DEFAULT 0,
  `salary` decimal(10,2) DEFAULT NULL,
  `hire_date` date NOT NULL,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`staff_id`),
  UNIQUE KEY `employee_id` (`employee_id`),
  KEY `user_id` (`user_id`),
  KEY `idx_staff_name` (`first_name`,`last_name`),
  KEY `idx_is_active` (`is_active`),
  CONSTRAINT `staff_ibfk_1` FOREIGN KEY (`user_id`) REFERENCES `users` (`user_id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول المرضى (Patients Table)
-- ===================================
CREATE TABLE `patients` (
  `patient_id` int(11) NOT NULL AUTO_INCREMENT,
  `patient_number` varchar(20) NOT NULL,
  `first_name` varchar(50) NOT NULL,
  `last_name` varchar(50) NOT NULL,
  `date_of_birth` date NOT NULL,
  `gender` enum('male','female') NOT NULL,
  `phone` varchar(20) NOT NULL,
  `email` varchar(100) DEFAULT NULL,
  `address` text DEFAULT NULL,
  `emergency_contact_name` varchar(100) DEFAULT NULL,
  `emergency_contact_phone` varchar(20) DEFAULT NULL,
  `medical_history` text DEFAULT NULL,
  `allergies` text DEFAULT NULL,
  `current_medications` text DEFAULT NULL,
  `chronic_conditions` text DEFAULT NULL,
  `insurance_info` text DEFAULT NULL,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`patient_id`),
  UNIQUE KEY `patient_number` (`patient_number`),
  KEY `idx_patients_phone` (`phone`),
  KEY `idx_patients_name` (`first_name`,`last_name`),
  KEY `idx_is_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول أنواع الإجراءات (Procedure Types Table)
-- ===================================
CREATE TABLE `procedure_types` (
  `procedure_type_id` int(11) NOT NULL AUTO_INCREMENT,
  `type_name` varchar(100) NOT NULL,
  `type_name_ar` varchar(100) NOT NULL,
  `description` text DEFAULT NULL,
  `description_ar` text DEFAULT NULL,
  `default_cost` decimal(10,2) DEFAULT 0.00,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`procedure_type_id`),
  KEY `idx_is_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول الإجراءات (Procedures Table)
-- ===================================
CREATE TABLE `procedures` (
  `procedure_id` int(11) NOT NULL AUTO_INCREMENT,
  `patient_id` int(11) NOT NULL,
  `staff_id` int(11) NOT NULL,
  `procedure_type_id` int(11) NOT NULL,
  `procedure_date` date NOT NULL,
  `tooth_involved` varchar(50) DEFAULT NULL,
  `description` text DEFAULT NULL,
  `notes` text DEFAULT NULL,
  `amount` decimal(10,2) NOT NULL,
  `additional_charges` decimal(10,2) DEFAULT 0.00,
  `discount` decimal(10,2) DEFAULT 0.00,
  `total_amount` decimal(10,2) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`procedure_id`),
  KEY `patient_id` (`patient_id`),
  KEY `staff_id` (`staff_id`),
  KEY `procedure_type_id` (`procedure_type_id`),
  KEY `idx_procedures_date` (`procedure_date`),
  CONSTRAINT `procedures_ibfk_1` FOREIGN KEY (`patient_id`) REFERENCES `patients` (`patient_id`) ON DELETE CASCADE,
  CONSTRAINT `procedures_ibfk_2` FOREIGN KEY (`staff_id`) REFERENCES `staff` (`staff_id`) ON DELETE RESTRICT,
  CONSTRAINT `procedures_ibfk_3` FOREIGN KEY (`procedure_type_id`) REFERENCES `procedure_types` (`procedure_type_id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول المدفوعات (Payments Table)
-- ===================================
CREATE TABLE `payments` (
  `payment_id` int(11) NOT NULL AUTO_INCREMENT,
  `procedure_id` int(11) NOT NULL,
  `payment_date` date NOT NULL,
  `amount_paid` decimal(10,2) NOT NULL,
  `payment_method` enum('cash','card','bank_transfer','insurance') NOT NULL,
  `payment_reference` varchar(100) DEFAULT NULL,
  `notes` text DEFAULT NULL,
  `received_by` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`payment_id`),
  KEY `procedure_id` (`procedure_id`),
  KEY `received_by` (`received_by`),
  KEY `idx_payments_date` (`payment_date`),
  CONSTRAINT `payments_ibfk_1` FOREIGN KEY (`procedure_id`) REFERENCES `procedures` (`procedure_id`) ON DELETE CASCADE,
  CONSTRAINT `payments_ibfk_2` FOREIGN KEY (`received_by`) REFERENCES `staff` (`staff_id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول الفواتير (Invoices Table)
-- ===================================
CREATE TABLE `invoices` (
  `invoice_id` int(11) NOT NULL AUTO_INCREMENT,
  `invoice_number` varchar(20) NOT NULL,
  `patient_id` int(11) NOT NULL,
  `invoice_date` date NOT NULL,
  `due_date` date DEFAULT NULL,
  `subtotal` decimal(10,2) NOT NULL,
  `tax_amount` decimal(10,2) DEFAULT 0.00,
  `discount_amount` decimal(10,2) DEFAULT 0.00,
  `total_amount` decimal(10,2) NOT NULL,
  `paid_amount` decimal(10,2) DEFAULT 0.00,
  `balance_due` decimal(10,2) NOT NULL,
  `status` enum('draft','sent','paid','overdue','cancelled') DEFAULT 'draft',
  `notes` text DEFAULT NULL,
  `created_by` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`invoice_id`),
  UNIQUE KEY `invoice_number` (`invoice_number`),
  KEY `patient_id` (`patient_id`),
  KEY `created_by` (`created_by`),
  KEY `idx_invoices_date` (`invoice_date`),
  KEY `idx_status` (`status`),
  CONSTRAINT `invoices_ibfk_1` FOREIGN KEY (`patient_id`) REFERENCES `patients` (`patient_id`) ON DELETE CASCADE,
  CONSTRAINT `invoices_ibfk_2` FOREIGN KEY (`created_by`) REFERENCES `staff` (`staff_id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول عناصر الفاتورة (Invoice Items Table)
-- ===================================
CREATE TABLE `invoice_items` (
  `item_id` int(11) NOT NULL AUTO_INCREMENT,
  `invoice_id` int(11) NOT NULL,
  `procedure_id` int(11) NOT NULL,
  `description` text NOT NULL,
  `quantity` int(11) DEFAULT 1,
  `unit_price` decimal(10,2) NOT NULL,
  `total_price` decimal(10,2) NOT NULL,
  PRIMARY KEY (`item_id`),
  KEY `invoice_id` (`invoice_id`),
  KEY `procedure_id` (`procedure_id`),
  CONSTRAINT `invoice_items_ibfk_1` FOREIGN KEY (`invoice_id`) REFERENCES `invoices` (`invoice_id`) ON DELETE CASCADE,
  CONSTRAINT `invoice_items_ibfk_2` FOREIGN KEY (`procedure_id`) REFERENCES `procedures` (`procedure_id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول المواعيد (Appointments Table)
-- ===================================
CREATE TABLE `appointments` (
  `appointment_id` int(11) NOT NULL AUTO_INCREMENT,
  `patient_id` int(11) NOT NULL,
  `staff_id` int(11) NOT NULL,
  `appointment_date` date NOT NULL,
  `appointment_time` time NOT NULL,
  `duration_minutes` int(11) DEFAULT 30,
  `procedure_type_id` int(11) DEFAULT NULL,
  `status` enum('scheduled','confirmed','completed','cancelled','no_show') DEFAULT 'scheduled',
  `notes` text DEFAULT NULL,
  `reminder_sent` tinyint(1) DEFAULT 0,
  `created_by` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`appointment_id`),
  KEY `patient_id` (`patient_id`),
  KEY `staff_id` (`staff_id`),
  KEY `procedure_type_id` (`procedure_type_id`),
  KEY `created_by` (`created_by`),
  KEY `idx_appointments_date` (`appointment_date`),
  KEY `idx_status` (`status`),
  CONSTRAINT `appointments_ibfk_1` FOREIGN KEY (`patient_id`) REFERENCES `patients` (`patient_id`) ON DELETE CASCADE,
  CONSTRAINT `appointments_ibfk_2` FOREIGN KEY (`staff_id`) REFERENCES `staff` (`staff_id`) ON DELETE RESTRICT,
  CONSTRAINT `appointments_ibfk_3` FOREIGN KEY (`procedure_type_id`) REFERENCES `procedure_types` (`procedure_type_id`) ON DELETE SET NULL,
  CONSTRAINT `appointments_ibfk_4` FOREIGN KEY (`created_by`) REFERENCES `users` (`user_id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول سجل تسجيل الدخول (Login Logs Table)
-- ===================================
CREATE TABLE `login_logs` (
  `log_id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) NOT NULL,
  `login_time` timestamp NOT NULL DEFAULT current_timestamp(),
  `ip_address` varchar(45) DEFAULT NULL,
  `user_agent` text DEFAULT NULL,
  `login_status` enum('success','failed') NOT NULL,
  `failure_reason` varchar(100) DEFAULT NULL,
  PRIMARY KEY (`log_id`),
  KEY `user_id` (`user_id`),
  KEY `idx_login_time` (`login_time`),
  KEY `idx_login_status` (`login_status`),
  CONSTRAINT `login_logs_ibfk_1` FOREIGN KEY (`user_id`) REFERENCES `users` (`user_id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول سجل الأنشطة (Activity Logs Table)
-- ===================================
CREATE TABLE `activity_logs` (
  `activity_id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) NOT NULL,
  `action` varchar(100) NOT NULL,
  `details` text DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`activity_id`),
  KEY `user_id` (`user_id`),
  KEY `idx_created_at` (`created_at`),
  CONSTRAINT `activity_logs_ibfk_1` FOREIGN KEY (`user_id`) REFERENCES `users` (`user_id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول النسخ الاحتياطية (Backups Table)
-- ===================================
CREATE TABLE `backups` (
  `backup_id` int(11) NOT NULL AUTO_INCREMENT,
  `backup_name` varchar(100) NOT NULL,
  `backup_path` varchar(255) NOT NULL,
  `backup_size` bigint(20) DEFAULT NULL,
  `backup_type` enum('manual','automatic') NOT NULL,
  `created_by` int(11) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`backup_id`),
  KEY `created_by` (`created_by`),
  KEY `idx_created_at` (`created_at`),
  CONSTRAINT `backups_ibfk_1` FOREIGN KEY (`created_by`) REFERENCES `users` (`user_id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول الإعدادات (Settings Table)
-- ===================================
CREATE TABLE `settings` (
  `setting_id` int(11) NOT NULL AUTO_INCREMENT,
  `setting_key` varchar(100) NOT NULL,
  `setting_value` text DEFAULT NULL,
  `setting_type` enum('string','number','boolean','json') DEFAULT 'string',
  `description` text DEFAULT NULL,
  `is_public` tinyint(1) DEFAULT 0,
  `updated_by` int(11) DEFAULT NULL,
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`setting_id`),
  UNIQUE KEY `setting_key` (`setting_key`),
  KEY `updated_by` (`updated_by`),
  CONSTRAINT `settings_ibfk_1` FOREIGN KEY (`updated_by`) REFERENCES `users` (`user_id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- جدول الإشعارات (Notifications Table)
-- ===================================
CREATE TABLE `notifications` (
  `notification_id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) NOT NULL,
  `title` varchar(200) NOT NULL,
  `title_ar` varchar(200) NOT NULL,
  `message` text NOT NULL,
  `message_ar` text NOT NULL,
  `type` enum('info','warning','error','success') DEFAULT 'info',
  `is_read` tinyint(1) DEFAULT 0,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`notification_id`),
  KEY `user_id` (`user_id`),
  KEY `idx_is_read` (`is_read`),
  KEY `idx_created_at` (`created_at`),
  CONSTRAINT `notifications_ibfk_1` FOREIGN KEY (`user_id`) REFERENCES `users` (`user_id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================
-- إدراج البيانات الأولية
-- Initial Data Insertion
-- ===================================

-- إدراج أنواع الإجراءات الأساسية
INSERT INTO `procedure_types` (`type_name`, `type_name_ar`, `description`, `description_ar`, `default_cost`) VALUES
('Dental Cleaning', 'تنظيف الأسنان', 'Regular dental cleaning and polishing', 'تنظيف وتلميع الأسنان العادي', 150.00),
('Tooth Filling', 'حشو الأسنان', 'Dental filling for cavities', 'حشو الأسنان للتسوس', 200.00),
('Tooth Extraction', 'خلع الأسنان', 'Tooth extraction procedure', 'إجراء خلع الأسنان', 300.00),
('Root Canal Treatment', 'علاج جذور الأسنان', 'Root canal therapy', 'علاج قناة الجذر', 800.00),
('Dental Crown', 'تاج الأسنان', 'Dental crown placement', 'وضع تاج الأسنان', 1200.00),
('Teeth Whitening', 'تبييض الأسنان', 'Professional teeth whitening', 'تبييض الأسنان المهني', 500.00),
('Dental Implant', 'زراعة الأسنان', 'Dental implant surgery', 'جراحة زراعة الأسنان', 3000.00),
('Orthodontic Treatment', 'علاج تقويم الأسنان', 'Braces and orthodontic care', 'تقويم الأسنان والرعاية التقويمية', 5000.00),
('Dental Bridge', 'جسر الأسنان', 'Dental bridge installation', 'تركيب جسر الأسنان', 2000.00),
('Gum Treatment', 'علاج اللثة', 'Periodontal treatment', 'علاج أمراض اللثة', 400.00);

-- إدراج المستخدم الإداري الافتراضي
INSERT INTO `users` (`username`, `email`, `password_hash`, `first_name`, `last_name`, `user_type`) VALUES
('admin', 'admin@dentalclinic.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'مدير', 'النظام', 'admin');

-- إدراج الإعدادات الأساسية
INSERT INTO `settings` (`setting_key`, `setting_value`, `setting_type`, `description`, `is_public`) VALUES
('clinic_name', 'عيادة الأسنان المتخصصة', 'string', 'اسم العيادة', 1),
('clinic_name_en', 'Specialized Dental Clinic', 'string', 'اسم العيادة بالإنجليزية', 1),
('clinic_address', 'الرياض، المملكة العربية السعودية', 'string', 'عنوان العيادة', 1),
('clinic_phone', '+966501234567', 'string', 'رقم هاتف العيادة', 1),
('clinic_email', 'info@dentalclinic.com', 'string', 'بريد العيادة الإلكتروني', 1),
('working_hours', 'السبت - الخميس: 8:00 ص - 10:00 م', 'string', 'ساعات العمل', 1),
('working_hours_en', 'Saturday - Thursday: 8:00 AM - 10:00 PM', 'string', 'ساعات العمل بالإنجليزية', 1),
('tax_rate', '15', 'number', 'معدل الضريبة المضافة (%)', 0),
('currency', 'ريال سعودي', 'string', 'العملة المستخدمة', 1),
('currency_en', 'Saudi Riyal', 'string', 'العملة المستخدمة بالإنجليزية', 1),
('appointment_duration', '30', 'number', 'مدة الموعد الافتراضية (دقيقة)', 0),
('backup_enabled', '1', 'boolean', 'تفعيل النسخ الاحتياطي التلقائي', 0),
('notification_enabled', '1', 'boolean', 'تفعيل الإشعارات', 0);

COMMIT;

-- ===================================
-- إنشاء المشاهدات (Views) المفيدة
-- ===================================

-- مشاهدة تفاصيل الإجراءات مع أسماء المرضى والموظفين
CREATE VIEW `procedure_details` AS
SELECT 
    p.procedure_id,
    p.procedure_date,
    CONCAT(pt.first_name, ' ', pt.last_name) AS patient_name,
    pt.patient_number,
    CONCAT(s.first_name, ' ', s.last_name) AS staff_name,
    s.employee_id,
    ptype.type_name_ar AS procedure_type,
    p.tooth_involved,
    p.description,
    p.amount,
    p.additional_charges,
    p.discount,
    p.total_amount,
    COALESCE(SUM(pay.amount_paid), 0) AS total_paid,
    (p.total_amount - COALESCE(SUM(pay.amount_paid), 0)) AS balance_due
FROM procedures p
JOIN patients pt ON p.patient_id = pt.patient_id
JOIN staff s ON p.staff_id = s.staff_id
JOIN procedure_types ptype ON p.procedure_type_id = ptype.procedure_type_id
LEFT JOIN payments pay ON p.procedure_id = pay.procedure_id
GROUP BY p.procedure_id;

-- مشاهدة إحصائيات المرضى
CREATE VIEW `patient_statistics` AS
SELECT 
    p.patient_id,
    CONCAT(p.first_name, ' ', p.last_name) AS patient_name,
    p.patient_number,
    p.phone,
    COUNT(pr.procedure_id) AS total_procedures,
    COALESCE(SUM(pr.total_amount), 0) AS total_amount,
    COALESCE(SUM(pay.amount_paid), 0) AS total_paid,
    (COALESCE(SUM(pr.total_amount), 0) - COALESCE(SUM(pay.amount_paid), 0)) AS balance_due,
    MAX(pr.procedure_date) AS last_visit
FROM patients p
LEFT JOIN procedures pr ON p.patient_id = pr.patient_id
LEFT JOIN payments pay ON pr.procedure_id = pay.procedure_id
GROUP BY p.patient_id;

-- مشاهدة إحصائيات الموظفين
CREATE VIEW `staff_statistics` AS
SELECT 
    s.staff_id,
    CONCAT(s.first_name, ' ', s.last_name) AS staff_name,
    s.employee_id,
    s.specialization,
    COUNT(pr.procedure_id) AS total_procedures,
    COALESCE(SUM(pr.total_amount), 0) AS total_revenue,
    COUNT(DISTINCT pr.patient_id) AS unique_patients,
    MAX(pr.procedure_date) AS last_procedure_date
FROM staff s
LEFT JOIN procedures pr ON s.staff_id = pr.staff_id
GROUP BY s.staff_id;

-- ===================================
-- إنشاء المؤشرات الإضافية لتحسين الأداء
-- ===================================

-- مؤشرات مركبة للبحث المتقدم
CREATE INDEX idx_patient_search ON patients(first_name, last_name, phone, patient_number);
CREATE INDEX idx_staff_search ON staff(first_name, last_name, employee_id, specialization);
CREATE INDEX idx_procedure_search ON procedures(procedure_date, patient_id, staff_id);
CREATE INDEX idx_payment_search ON payments(payment_date, procedure_id, payment_method);
CREATE INDEX idx_appointment_search ON appointments(appointment_date, patient_id, staff_id, status);

-- مؤشرات للتقارير
CREATE INDEX idx_procedures_monthly ON procedures(YEAR(procedure_date), MONTH(procedure_date));
CREATE INDEX idx_payments_monthly ON payments(YEAR(payment_date), MONTH(payment_date));
CREATE INDEX idx_invoices_monthly ON invoices(YEAR(invoice_date), MONTH(invoice_date));

-- ===================================
-- إنشاء المحفزات (Triggers) للحفاظ على سلامة البيانات
-- ===================================

-- محفز لتحديث رصيد الفاتورة عند إضافة دفعة جديدة
DELIMITER $$
CREATE TRIGGER update_invoice_balance_after_payment
AFTER INSERT ON payments
FOR EACH ROW
BEGIN
    DECLARE invoice_id_var INT;
    DECLARE total_paid DECIMAL(10,2);
    
    -- الحصول على رقم الفاتورة من الإجراء
    SELECT i.invoice_id INTO invoice_id_var
    FROM invoices i
    JOIN invoice_items ii ON i.invoice_id = ii.invoice_id
    WHERE ii.procedure_id = NEW.procedure_id
    LIMIT 1;
    
    -- إذا وجدت فاتورة، قم بتحديث الرصيد
    IF invoice_id_var IS NOT NULL THEN
        -- حساب إجمالي المدفوع للفاتورة
        SELECT COALESCE(SUM(p.amount_paid), 0) INTO total_paid
        FROM payments p
        JOIN procedures pr ON p.procedure_id = pr.procedure_id
        JOIN invoice_items ii ON pr.procedure_id = ii.procedure_id
        WHERE ii.invoice_id = invoice_id_var;
        
        -- تحديث الفاتورة
        UPDATE invoices 
        SET paid_amount = total_paid,
            balance_due = total_amount - total_paid,
            status = CASE 
                WHEN total_paid >= total_amount THEN 'paid'
                WHEN total_paid > 0 THEN 'sent'
                ELSE status
            END
        WHERE invoice_id = invoice_id_var;
    END IF;
END$$
DELIMITER ;

-- محفز لإنشاء رقم مريض تلقائي
DELIMITER $$
CREATE TRIGGER generate_patient_number
BEFORE INSERT ON patients
FOR EACH ROW
BEGIN
    DECLARE next_number INT;
    DECLARE new_patient_number VARCHAR(20);
    
    -- الحصول على آخر رقم مريض
    SELECT COALESCE(MAX(CAST(SUBSTRING(patient_number, 2) AS UNSIGNED)), 0) + 1 
    INTO next_number 
    FROM patients 
    WHERE patient_number REGEXP '^P[0-9]+$';
    
    -- إنشاء رقم المريض الجديد
    SET new_patient_number = CONCAT('P', LPAD(next_number, 6, '0'));
    
    -- تعيين الرقم الجديد
    SET NEW.patient_number = new_patient_number;
END$$
DELIMITER ;

-- محفز لإنشاء رقم موظف تلقائي
DELIMITER $$
CREATE TRIGGER generate_employee_id
BEFORE INSERT ON staff
FOR EACH ROW
BEGIN
    DECLARE next_number INT;
    DECLARE new_employee_id VARCHAR(20);
    
    -- الحصول على آخر رقم موظف
    SELECT COALESCE(MAX(CAST(SUBSTRING(employee_id, 2) AS UNSIGNED)), 0) + 1 
    INTO next_number 
    FROM staff 
    WHERE employee_id REGEXP '^E[0-9]+$';
    
    -- إنشاء رقم الموظف الجديد
    SET new_employee_id = CONCAT('E', LPAD(next_number, 4, '0'));
    
    -- تعيين الرقم الجديد
    SET NEW.employee_id = new_employee_id;
END$$
DELIMITER ;

-- محفز لإنشاء رقم فاتورة تلقائي
DELIMITER $$
CREATE TRIGGER generate_invoice_number
BEFORE INSERT ON invoices
FOR EACH ROW
BEGIN
    DECLARE next_number INT;
    DECLARE new_invoice_number VARCHAR(20);
    DECLARE current_year VARCHAR(4);
    
    SET current_year = YEAR(CURDATE());
    
    -- الحصول على آخر رقم فاتورة للسنة الحالية
    SELECT COALESCE(MAX(CAST(SUBSTRING(invoice_number, 6) AS UNSIGNED)), 0) + 1 
    INTO next_number 
    FROM invoices 
    WHERE invoice_number LIKE CONCAT(current_year, '-%');
    
    -- إنشاء رقم الفاتورة الجديد
    SET new_invoice_number = CONCAT(current_year, '-', LPAD(next_number, 6, '0'));
    
    -- تعيين الرقم الجديد
    SET NEW.invoice_number = new_invoice_number;
END$$
DELIMITER ;

-- ===================================
-- إنشاء الإجراءات المخزنة (Stored Procedures)
-- ===================================

-- إجراء للحصول على إحصائيات شهرية
DELIMITER $$
CREATE PROCEDURE GetMonthlyStatistics(IN target_year INT, IN target_month INT)
BEGIN
    SELECT 
        'الإجراءات' as category,
        COUNT(*) as count,
        COALESCE(SUM(total_amount), 0) as total_amount
    FROM procedures 
    WHERE YEAR(procedure_date) = target_year AND MONTH(procedure_date) = target_month
    
    UNION ALL
    
    SELECT 
        'المدفوعات' as category,
        COUNT(*) as count,
        COALESCE(SUM(amount_paid), 0) as total_amount
    FROM payments 
    WHERE YEAR(payment_date) = target_year AND MONTH(payment_date) = target_month
    
    UNION ALL
    
    SELECT 
        'المرضى الجدد' as category,
        COUNT(*) as count,
        0 as total_amount
    FROM patients 
    WHERE YEAR(created_at) = target_year AND MONTH(created_at) = target_month;
END$$
DELIMITER ;

-- إجراء للحصول على تقرير المريض الشامل
DELIMITER $$
CREATE PROCEDURE GetPatientReport(IN patient_id_param INT)
BEGIN
    -- معلومات المريض الأساسية
    SELECT * FROM patients WHERE patient_id = patient_id_param;
    
    -- إجراءات المريض
    SELECT * FROM procedure_details WHERE patient_id = patient_id_param ORDER BY procedure_date DESC;
    
    -- مواعيد المريض
    SELECT 
        a.*,
        CONCAT(s.first_name, ' ', s.last_name) as doctor_name,
        pt.type_name_ar as procedure_type
    FROM appointments a
    JOIN staff s ON a.staff_id = s.staff_id
    LEFT JOIN procedure_types pt ON a.procedure_type_id = pt.procedure_type_id
    WHERE a.patient_id = patient_id_param
    ORDER BY a.appointment_date DESC, a.appointment_time DESC;
    
    -- ملخص مالي
    SELECT 
        COUNT(pr.procedure_id) as total_procedures,
        COALESCE(SUM(pr.total_amount), 0) as total_amount,
        COALESCE(SUM(pay.amount_paid), 0) as total_paid,
        (COALESCE(SUM(pr.total_amount), 0) - COALESCE(SUM(pay.amount_paid), 0)) as balance_due
    FROM procedures pr
    LEFT JOIN payments pay ON pr.procedure_id = pay.procedure_id
    WHERE pr.patient_id = patient_id_param;
END$$
DELIMITER ;

DELIMITER ;

-- انتهاء إنشاء قاعدة البيانات
-- Database creation completed

