<?php
/**
 * ملف الاتصال بقاعدة البيانات
 * Database Connection File
 */

// تضمين ملف الإعدادات
require_once __DIR__ . '/../config/database.php';

class DatabaseConnection {
    private static $instance = null;
    private $connection;
    
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];
            
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            if (DEBUG_MODE) {
                die("خطأ في الاتصال بقاعدة البيانات: " . $e->getMessage());
            } else {
                die("خطأ في الاتصال بقاعدة البيانات");
            }
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->connection;
    }
    
    // منع استنساخ الكائن
    private function __clone() {}
    
    // منع إلغاء تسلسل الكائن
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}

// دالة للحصول على اتصال قاعدة البيانات
function getDBConnection() {
    return DatabaseConnection::getInstance()->getConnection();
}

// دالة لتنفيذ استعلام SELECT
function executeQuery($sql, $params = []) {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        logError("Database Query Error: " . $e->getMessage() . " | SQL: " . $sql);
        if (DEBUG_MODE) {
            throw $e;
        }
        return false;
    }
}

// دالة لتنفيذ استعلام INSERT/UPDATE/DELETE
function executeNonQuery($sql, $params = []) {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);
        return $result ? $stmt->rowCount() : false;
    } catch (PDOException $e) {
        logError("Database Non-Query Error: " . $e->getMessage() . " | SQL: " . $sql);
        if (DEBUG_MODE) {
            throw $e;
        }
        return false;
    }
}

// دالة للحصول على آخر ID مدرج
function getLastInsertId() {
    try {
        $pdo = getDBConnection();
        return $pdo->lastInsertId();
    } catch (PDOException $e) {
        logError("Get Last Insert ID Error: " . $e->getMessage());
        return false;
    }
}

// دالة لبدء معاملة قاعدة البيانات
function beginTransaction() {
    try {
        $pdo = getDBConnection();
        return $pdo->beginTransaction();
    } catch (PDOException $e) {
        logError("Begin Transaction Error: " . $e->getMessage());
        return false;
    }
}

// دالة لتأكيد المعاملة
function commitTransaction() {
    try {
        $pdo = getDBConnection();
        return $pdo->commit();
    } catch (PDOException $e) {
        logError("Commit Transaction Error: " . $e->getMessage());
        return false;
    }
}

// دالة لإلغاء المعاملة
function rollbackTransaction() {
    try {
        $pdo = getDBConnection();
        return $pdo->rollback();
    } catch (PDOException $e) {
        logError("Rollback Transaction Error: " . $e->getMessage());
        return false;
    }
}

// دالة لتسجيل الأخطاء
function logError($message) {
    if (LOG_ERRORS) {
        $timestamp = date(DATETIME_FORMAT);
        $logMessage = "[$timestamp] $message" . PHP_EOL;
        file_put_contents(ERROR_LOG_PATH, $logMessage, FILE_APPEND | LOCK_EX);
    }
}

// اختبار الاتصال عند تحميل الملف
try {
    $testConnection = getDBConnection();
    if (DEBUG_MODE) {
        // echo "تم الاتصال بقاعدة البيانات بنجاح";
    }
} catch (Exception $e) {
    if (DEBUG_MODE) {
        echo "فشل في الاتصال بقاعدة البيانات: " . $e->getMessage();
    }
}
?>

