<?php
/**
 * ملف الوظائف المساعدة العامة
 * General Helper Functions File
 */

// تضمين ملفات الإعداد والاتصال
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/connection.php';

// بدء الجلسة إذا لم تكن مبدوءة
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * وظائف الأمان والمصادقة
 * Security and Authentication Functions
 */

// دالة تشفير كلمة المرور
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

// دالة التحقق من كلمة المرور
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

// دالة تنظيف البيانات المدخلة
function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

// دالة التحقق من صحة البريد الإلكتروني
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// دالة التحقق من قوة كلمة المرور
function validatePassword($password) {
    if (strlen($password) < PASSWORD_MIN_LENGTH) {
        return false;
    }
    // يجب أن تحتوي على حرف كبير وصغير ورقم
    if (!preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).+$/', $password)) {
        return false;
    }
    return true;
}

// دالة التحقق من رقم الهاتف
function validatePhone($phone) {
    // إزالة المسافات والرموز
    $phone = preg_replace('/[^0-9+]/', '', $phone);
    // التحقق من الطول والتنسيق
    return preg_match('/^(\+966|966|05)[0-9]{8}$/', $phone);
}

/**
 * وظائف إدارة الجلسات
 * Session Management Functions
 */

// دالة تسجيل دخول المستخدم
function loginUser($userId, $userType, $username) {
    $_SESSION['user_id'] = $userId;
    $_SESSION['user_type'] = $userType;
    $_SESSION['username'] = $username;
    $_SESSION['login_time'] = time();
    $_SESSION['last_activity'] = time();
    
    // تسجيل عملية تسجيل الدخول
    logUserLogin($userId, 'success');
}

// دالة تسجيل خروج المستخدم
function logoutUser() {
    session_unset();
    session_destroy();
    session_start();
}

// دالة التحقق من تسجيل دخول المستخدم
function isLoggedIn() {
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['login_time'])) {
        return false;
    }
    
    // التحقق من انتهاء صلاحية الجلسة
    if (time() - $_SESSION['last_activity'] > SESSION_TIMEOUT) {
        logoutUser();
        return false;
    }
    
    // تحديث وقت آخر نشاط
    $_SESSION['last_activity'] = time();
    return true;
}

// دالة التحقق من صلاحيات المستخدم
function hasPermission($requiredType) {
    if (!isLoggedIn()) {
        return false;
    }
    
    $userType = $_SESSION['user_type'];
    
    // المدير له صلاحيات كاملة
    if ($userType === 'admin') {
        return true;
    }
    
    // التحقق من الصلاحيات المحددة
    if (is_array($requiredType)) {
        return in_array($userType, $requiredType);
    }
    
    return $userType === $requiredType;
}

// دالة إعادة التوجيه للصفحة المناسبة حسب نوع المستخدم
function redirectToDashboard() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit;
    }
    
    $userType = $_SESSION['user_type'];
    switch ($userType) {
        case 'admin':
            header('Location: pages/admin/dashboard.php');
            break;
        case 'doctor':
            header('Location: pages/doctor/dashboard.php');
            break;
        case 'receptionist':
            header('Location: pages/receptionist/dashboard.php');
            break;
        default:
            header('Location: login.php');
    }
    exit;
}

/**
 * وظائف التاريخ والوقت
 * Date and Time Functions
 */

// دالة تنسيق التاريخ للعرض
function formatDate($date, $format = DISPLAY_DATE_FORMAT) {
    if (empty($date) || $date === '0000-00-00') {
        return '';
    }
    return date($format, strtotime($date));
}

// دالة تنسيق التاريخ والوقت للعرض
function formatDateTime($datetime, $format = DISPLAY_DATETIME_FORMAT) {
    if (empty($datetime) || $datetime === '0000-00-00 00:00:00') {
        return '';
    }
    return date($format, strtotime($datetime));
}

// دالة حساب العمر
function calculateAge($birthDate) {
    if (empty($birthDate) || $birthDate === '0000-00-00') {
        return 0;
    }
    
    $today = new DateTime();
    $birth = new DateTime($birthDate);
    $age = $today->diff($birth);
    
    return $age->y;
}

/**
 * وظائف التعامل مع الملفات
 * File Handling Functions
 */

// دالة رفع الملفات
function uploadFile($file, $allowedTypes = ['jpg', 'jpeg', 'png', 'pdf'], $maxSize = MAX_FILE_SIZE) {
    if (!isset($file['tmp_name']) || empty($file['tmp_name'])) {
        return ['success' => false, 'message' => 'لم يتم اختيار ملف'];
    }
    
    // التحقق من حجم الملف
    if ($file['size'] > $maxSize) {
        return ['success' => false, 'message' => 'حجم الملف كبير جداً'];
    }
    
    // التحقق من نوع الملف
    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($fileExtension, $allowedTypes)) {
        return ['success' => false, 'message' => 'نوع الملف غير مدعوم'];
    }
    
    // إنشاء اسم ملف فريد
    $fileName = uniqid() . '_' . time() . '.' . $fileExtension;
    $filePath = UPLOAD_PATH . $fileName;
    
    // رفع الملف
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
        return ['success' => true, 'filename' => $fileName, 'path' => $filePath];
    } else {
        return ['success' => false, 'message' => 'فشل في رفع الملف'];
    }
}

// دالة حذف الملف
function deleteFile($filename) {
    $filePath = UPLOAD_PATH . $filename;
    if (file_exists($filePath)) {
        return unlink($filePath);
    }
    return false;
}

/**
 * وظائف التعامل مع اللغات
 * Language Functions
 */

// دالة الحصول على اللغة الحالية
function getCurrentLanguage() {
    if (isset($_SESSION['language']) && in_array($_SESSION['language'], SUPPORTED_LANGUAGES)) {
        return $_SESSION['language'];
    }
    return DEFAULT_LANGUAGE;
}

// دالة تغيير اللغة
function setLanguage($language) {
    if (in_array($language, SUPPORTED_LANGUAGES)) {
        $_SESSION['language'] = $language;
        return true;
    }
    return false;
}

// دالة الحصول على النص المترجم
function getText($key, $language = null) {
    if ($language === null) {
        $language = getCurrentLanguage();
    }
    
    $langFile = __DIR__ . "/../languages/{$language}.php";
    if (file_exists($langFile)) {
        $translations = include $langFile;
        return isset($translations[$key]) ? $translations[$key] : $key;
    }
    
    return $key;
}

/**
 * وظائف التسجيل والمراقبة
 * Logging and Monitoring Functions
 */

// دالة تسجيل عملية تسجيل الدخول
function logUserLogin($userId, $status, $failureReason = null) {
    $sql = "INSERT INTO login_logs (user_id, login_time, ip_address, user_agent, login_status, failure_reason) 
            VALUES (?, NOW(), ?, ?, ?, ?)";
    
    $ipAddress = $_SERVER['REMOTE_ADDR'] ?? '';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    
    executeNonQuery($sql, [$userId, $ipAddress, $userAgent, $status, $failureReason]);
}

// دالة تسجيل العمليات المهمة
function logActivity($userId, $action, $details = null) {
    $sql = "INSERT INTO activity_logs (user_id, action, details, created_at) VALUES (?, ?, ?, NOW())";
    executeNonQuery($sql, [$userId, $action, $details]);
}

/**
 * وظائف التنسيق والعرض
 * Formatting and Display Functions
 */

// دالة تنسيق المبالغ المالية
function formatCurrency($amount, $currency = 'ريال') {
    return number_format($amount, 2) . ' ' . $currency;
}

// دالة تنسيق الأرقام
function formatNumber($number, $decimals = 0) {
    return number_format($number, $decimals);
}

// دالة إنشاء رقم فريد
function generateUniqueNumber($prefix = '', $length = 8) {
    $number = $prefix . str_pad(mt_rand(1, pow(10, $length) - 1), $length, '0', STR_PAD_LEFT);
    return $number;
}

// دالة إنشاء كلمة مرور عشوائية
function generateRandomPassword($length = 12) {
    $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*';
    $password = '';
    
    for ($i = 0; $i < $length; $i++) {
        $password .= $characters[rand(0, strlen($characters) - 1)];
    }
    
    return $password;
}

/**
 * وظائف التحقق من البيانات
 * Data Validation Functions
 */

// دالة التحقق من وجود المستخدم
function userExists($username, $email = null) {
    $sql = "SELECT COUNT(*) FROM users WHERE username = ?";
    $params = [$username];
    
    if ($email) {
        $sql .= " OR email = ?";
        $params[] = $email;
    }
    
    $stmt = executeQuery($sql, $params);
    return $stmt ? $stmt->fetchColumn() > 0 : false;
}

// دالة التحقق من وجود المريض
function patientExists($patientNumber, $phone = null) {
    $sql = "SELECT COUNT(*) FROM patients WHERE patient_number = ?";
    $params = [$patientNumber];
    
    if ($phone) {
        $sql .= " OR phone = ?";
        $params[] = $phone;
    }
    
    $stmt = executeQuery($sql, $params);
    return $stmt ? $stmt->fetchColumn() > 0 : false;
}

/**
 * وظائف الإشعارات
 * Notification Functions
 */

// دالة إضافة إشعار
function addNotification($userId, $title, $titleAr, $message, $messageAr, $type = 'info') {
    $sql = "INSERT INTO notifications (user_id, title, title_ar, message, message_ar, type, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW())";
    
    return executeNonQuery($sql, [$userId, $title, $titleAr, $message, $messageAr, $type]);
}

// دالة الحصول على الإشعارات غير المقروءة
function getUnreadNotifications($userId) {
    $sql = "SELECT * FROM notifications WHERE user_id = ? AND is_read = 0 ORDER BY created_at DESC";
    $stmt = executeQuery($sql, [$userId]);
    return $stmt ? $stmt->fetchAll() : [];
}

// دالة تمييز الإشعار كمقروء
function markNotificationAsRead($notificationId) {
    $sql = "UPDATE notifications SET is_read = 1 WHERE notification_id = ?";
    return executeNonQuery($sql, [$notificationId]);
}

/**
 * وظائف النسخ الاحتياطي
 * Backup Functions
 */

// دالة إنشاء نسخة احتياطية
function createBackup($type = 'manual') {
    $backupName = 'backup_' . date('Y-m-d_H-i-s') . '.sql';
    $backupPath = BACKUP_PATH . $backupName;
    
    $command = "mysqldump -h " . DB_HOST . " -u " . DB_USER . " -p" . DB_PASS . " " . DB_NAME . " > " . $backupPath;
    
    exec($command, $output, $returnCode);
    
    if ($returnCode === 0) {
        // تسجيل النسخة الاحتياطية في قاعدة البيانات
        $sql = "INSERT INTO backups (backup_name, backup_path, backup_size, backup_type, created_by, created_at) 
                VALUES (?, ?, ?, ?, ?, NOW())";
        
        $backupSize = file_exists($backupPath) ? filesize($backupPath) : 0;
        $createdBy = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;
        
        executeNonQuery($sql, [$backupName, $backupPath, $backupSize, $type, $createdBy]);
        
        return ['success' => true, 'filename' => $backupName];
    }
    
    return ['success' => false, 'message' => 'فشل في إنشاء النسخة الاحتياطية'];
}

// دالة استعادة النسخة الاحتياطية
function restoreBackup($backupPath) {
    if (!file_exists($backupPath)) {
        return ['success' => false, 'message' => 'ملف النسخة الاحتياطية غير موجود'];
    }
    
    $command = "mysql -h " . DB_HOST . " -u " . DB_USER . " -p" . DB_PASS . " " . DB_NAME . " < " . $backupPath;
    
    exec($command, $output, $returnCode);
    
    if ($returnCode === 0) {
        return ['success' => true, 'message' => 'تم استعادة النسخة الاحتياطية بنجاح'];
    }
    
    return ['success' => false, 'message' => 'فشل في استعادة النسخة الاحتياطية'];
}

/**
 * وظائف مساعدة أخرى
 * Other Helper Functions
 */

// دالة إعادة التوجيه مع رسالة
function redirectWithMessage($url, $message, $type = 'success') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
    header("Location: $url");
    exit;
}

// دالة عرض رسالة الفلاش
function displayFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'info';
        
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        
        $alertClass = 'alert-info';
        switch ($type) {
            case 'success':
                $alertClass = 'alert-success';
                break;
            case 'error':
                $alertClass = 'alert-danger';
                break;
            case 'warning':
                $alertClass = 'alert-warning';
                break;
        }
        
        echo "<div class='alert $alertClass alert-dismissible fade show' role='alert'>
                $message
                <button type='button' class='btn-close' data-bs-dismiss='alert'></button>
              </div>";
    }
}

// دالة التحقق من الصفحة النشطة
function isActivePage($page) {
    $currentPage = basename($_SERVER['PHP_SELF']);
    return $currentPage === $page ? 'active' : '';
}
?>

